#! /bin/bash

# Use the systfs interface to enable Virtual Functions on network
# adapters based on the old method of setting the max_vfs module option.

# Optional parameters
# -f	Force the setting of the number of VFs, even if VFs are laready enabled.
# -v	Be verbose about what is being done. Normally this script runs quitely
#	as a systemd service.

FORCE=''
VERBOSE=''

for arg; do
	case "$arg" in
		-f)
			FORCE=y
		;;
		-v)
			VERBOSE=y
		;;
		*)
			echo "\"$arg\" is not a valid option. It is ignored." >&2
		;;
	esac
done

# Given a space or comma separated list and a network device, find the position
# of the network device in its driver's list of devices and return the value in
# the list that is at that position. If there are not enough values in the
# list to correspond to the device's position in the driver's list, return the
# last value in the list.
#
# The list of devices in the driver list is ordered by the time each device
# entry was created. That assumes the devices were created in discovery order
# which would correspond to the order the devices would be given values from the
# list.
function get_indexed_max_vfs {

	# Convert to space separated list.
	local max_vfs_list=${1//,/ }
	local net=$2
	local IDX=0

	NET_PCI_LINK=$(readlink /sys/class/net/$net/device)
	NET_PCI=${NET_PCI_LINK##*/}

	NET_DRIVER_DIR=$(readlink -e /sys/class/net/$net/device/driver)
	pushd $NET_DRIVER_DIR >/dev/null

	# Get a list of the PCI devices, sorted by creation time.
	DEVICE_LIST=$(ls -1Sd 0000:*)
	for DEV in $DEVICE_LIST; do
		if [ "$DEV" = "$NET_PCI" ]; then
			break
		fi
		# The device is not at this index.
		# Pop the first value off the list.
		# The last value will always remain. If $net is at a position
		# greater than the number of values in the list, the last value
		# will be returned.
		if [ "${max_vfs_list/ /}" != "$max_vfs_list" ]; then
			max_vfs_list=${max_vfs_list#* }
			((IDX++))
		fi
	done
	popd >/dev/null

	# Return the first value and its position on the list.
	echo ${max_vfs_list%% *}
	return $IDX
}

pushd /sys/class/net >/dev/null
for NET in $(ls); do
	if [ ! -e $NET/device/sriov_numvfs ]; then
		if [ -n "$VERBOSE" ]; then
			echo "$NET has no VFs. Skipping."
		fi
		continue
	fi

	NUM_VFS=$(cat $NET/device/sriov_numvfs)

	# Saftey check in case the read fails.
	if [ -z "$NUM_VFS" ]; then
		if [ -n "$VERBOSE" ]; then
			echo "Error reading number of VFs for device $NET. Skipping."
		fi
		continue
	fi

	# Skip if VFs are already enabled.
	if [ $NUM_VFS -ne 0 ]; then
		if [ -z "$FORCE" ]; then
			if [ -n "$VERBOSE" ]; then
				echo "$NET already has $NUM_VFS VFs. Skipping. If this is the wrong value, try using the -f (force) option.)"
			fi
			continue
		fi
	fi

	# Get the driver for the interface.
	DRIVER_DIR=$(readlink $NET/device/driver)
	DRIVER=${DRIVER_DIR##*/}

	# Look for a config file in /etc/modprobe.d/ that has options for the driver.
	# If multiple entries are found, the last one is used.
	OPTIONS=$(grep '^options \+'$DRIVER' .*max_vfs' /etc/modprobe.d/* | tail -n1)
	if [ -z "$OPTIONS" ]; then
		if [ -n "$VERBOSE" ]; then
			echo "Did not find max_vfs option for driver $DRIVER in /etc/modprobe.d/ directory. Skipping $NET."
		fi
		continue
	fi

	MAX_VFS=$(echo "$OPTIONS" | sed -e 's/.* max_vfs=\([0-9,]\+\).*/\1/')
	if [ -z "$MAX_VFS" ]; then
		if [ -n "$VERBOSE" ]; then
			echo "Did not find a value for the max_vfs option for driver $DRIVER in /etc/modprobe.d/ directory. Skipping $NET."
		fi
		continue
	fi

	if [ "${MAX_VFS/,/}" != "$MAX_VFS" ]; then
		MAX_VFS=$(get_indexed_max_vfs $MAX_VFS $NET)
		IDX=$?
		if [ -n "$VERBOSE" ]; then
			echo "Using max_vfs value $MAX_VFS at index $IDX in the list."
		fi
	fi
	if [ $NUM_VFS -eq $MAX_VFS ]; then
		if [ -n "$VERBOSE" ]; then
			echo "$NET already has $NUM_VFS VFs. Skipping."
		fi
		continue
	fi
	if [ $NUM_VFS -ne 0 ]; then
		# We know that FORCE is set, else we would have bailed earlier.
		if [ -n "$VERBOSE" ]; then
			echo "Resetting the number of VFs on $NET. The existing $NUM_VFS VFs will be deleted."
		fi

		echo 0 >$NET/device/sriov_numvfs
	fi

	if [ -n "$VERBOSE" ]; then
		echo "Setting the number of VFs on $NET to $MAX_VFS."
	fi
	echo $MAX_VFS >$NET/device/sriov_numvfs
done
popd >/dev/null
